"""
Node.js project templates - Express, Next.js, Vite+React
"""
from typing import List, Dict
from ucts.templates.base import ProjectTemplate, TemplateFile, TemplateConfig
import json


class ExpressTemplate(ProjectTemplate):
    """Express.js REST API template"""

    @property
    def name(self) -> str:
        return "express"

    @property
    def display_name(self) -> str:
        return "Express.js"

    @property
    def description(self) -> str:
        return "Express.js REST API with TypeScript"

    @property
    def language(self) -> str:
        return "typescript"

    @property
    def tags(self) -> List[str]:
        return ["api", "rest", "node", "express"]

    @property
    def dependencies(self) -> Dict[str, List[str]]:
        return {
            "node": [
                "express",
                "cors",
                "helmet",
                "dotenv",
            ]
        }

    @property
    def dev_dependencies(self) -> Dict[str, List[str]]:
        return {
            "node": [
                "typescript",
                "@types/node",
                "@types/express",
                "@types/cors",
                "ts-node",
                "nodemon",
                "jest",
                "@types/jest",
                "ts-jest",
            ]
        }

    def generate(self, config: TemplateConfig) -> List[TemplateFile]:
        files = []

        # package.json
        package = {
            "name": config.project_name,
            "version": config.version,
            "description": config.description or "Express.js API",
            "main": "dist/index.js",
            "scripts": {
                "dev": "nodemon src/index.ts",
                "build": "tsc",
                "start": "node dist/index.js",
                "test": "jest",
            },
            "dependencies": {
                "express": "^4.18.0",
                "cors": "^2.8.0",
                "helmet": "^7.0.0",
                "dotenv": "^16.0.0",
            },
            "devDependencies": {
                "typescript": "^5.0.0",
                "@types/node": "^20.0.0",
                "@types/express": "^4.17.0",
                "@types/cors": "^2.8.0",
                "ts-node": "^10.0.0",
                "nodemon": "^3.0.0",
                "jest": "^29.0.0",
                "@types/jest": "^29.0.0",
                "ts-jest": "^29.0.0",
            }
        }
        files.append(TemplateFile(
            path="package.json",
            content=json.dumps(package, indent=2)
        ))

        # tsconfig.json
        tsconfig = {
            "compilerOptions": {
                "target": "ES2022",
                "module": "commonjs",
                "lib": ["ES2022"],
                "outDir": "./dist",
                "rootDir": "./src",
                "strict": True,
                "esModuleInterop": True,
                "skipLibCheck": True,
                "forceConsistentCasingInFileNames": True,
                "resolveJsonModule": True
            },
            "include": ["src/**/*"],
            "exclude": ["node_modules", "dist"]
        }
        files.append(TemplateFile(
            path="tsconfig.json",
            content=json.dumps(tsconfig, indent=2)
        ))

        # src/index.ts
        files.append(TemplateFile(
            path="src/index.ts",
            content=f'''import express from "express";
import cors from "cors";
import helmet from "helmet";
import dotenv from "dotenv";
import {{ router }} from "./routes";

dotenv.config();

const app = express();
const PORT = process.env.PORT || 3000;

// Middleware
app.use(helmet());
app.use(cors());
app.use(express.json());

// Routes
app.use("/api", router);

// Health check
app.get("/health", (req, res) => {{
  res.json({{ status: "healthy", version: "{config.version}" }});
}});

app.listen(PORT, () => {{
  console.log(`Server running on port ${{PORT}}`);
}});

export {{ app }};
'''
        ))

        # src/routes/index.ts
        files.append(TemplateFile(
            path="src/routes/index.ts",
            content='''import { Router } from "express";

const router = Router();

router.get("/", (req, res) => {
  res.json({ message: "Welcome to the API" });
});

export { router };
'''
        ))

        # .env.example
        files.append(TemplateFile(
            path=".env.example",
            content='PORT=3000\nNODE_ENV=development\n'
        ))

        # nodemon.json
        files.append(TemplateFile(
            path="nodemon.json",
            content=json.dumps({
                "watch": ["src"],
                "ext": "ts",
                "exec": "ts-node src/index.ts"
            }, indent=2)
        ))

        # README.md
        files.append(TemplateFile(
            path="README.md",
            content=f'''# {config.project_name}

{config.description or "Express.js API"}

## Quick Start

```bash
npm install
npm run dev
```

## Scripts

- `npm run dev` - Start development server
- `npm run build` - Build for production
- `npm start` - Start production server
- `npm test` - Run tests
'''
        ))

        return files


class NextJSTemplate(ProjectTemplate):
    """Next.js application template"""

    @property
    def name(self) -> str:
        return "nextjs"

    @property
    def display_name(self) -> str:
        return "Next.js"

    @property
    def description(self) -> str:
        return "Next.js 14 with App Router and TypeScript"

    @property
    def language(self) -> str:
        return "typescript"

    @property
    def tags(self) -> List[str]:
        return ["web", "react", "ssr", "fullstack"]

    def generate(self, config: TemplateConfig) -> List[TemplateFile]:
        files = []

        # package.json
        package = {
            "name": config.project_name,
            "version": config.version,
            "private": True,
            "scripts": {
                "dev": "next dev",
                "build": "next build",
                "start": "next start",
                "lint": "next lint",
            },
            "dependencies": {
                "next": "^14.0.0",
                "react": "^18.0.0",
                "react-dom": "^18.0.0",
            },
            "devDependencies": {
                "typescript": "^5.0.0",
                "@types/node": "^20.0.0",
                "@types/react": "^18.0.0",
                "@types/react-dom": "^18.0.0",
                "eslint": "^8.0.0",
                "eslint-config-next": "^14.0.0",
            }
        }
        files.append(TemplateFile(
            path="package.json",
            content=json.dumps(package, indent=2)
        ))

        # tsconfig.json
        tsconfig = {
            "compilerOptions": {
                "lib": ["dom", "dom.iterable", "esnext"],
                "allowJs": True,
                "skipLibCheck": True,
                "strict": True,
                "noEmit": True,
                "esModuleInterop": True,
                "module": "esnext",
                "moduleResolution": "bundler",
                "resolveJsonModule": True,
                "isolatedModules": True,
                "jsx": "preserve",
                "incremental": True,
                "plugins": [{"name": "next"}],
                "paths": {"@/*": ["./*"]}
            },
            "include": ["next-env.d.ts", "**/*.ts", "**/*.tsx", ".next/types/**/*.ts"],
            "exclude": ["node_modules"]
        }
        files.append(TemplateFile(
            path="tsconfig.json",
            content=json.dumps(tsconfig, indent=2)
        ))

        # next.config.js
        files.append(TemplateFile(
            path="next.config.js",
            content='/** @type {import("next").NextConfig} */\nmodule.exports = {\n  reactStrictMode: true,\n};\n'
        ))

        # app/layout.tsx
        files.append(TemplateFile(
            path="app/layout.tsx",
            content=f'''import type {{ Metadata }} from "next";
import "./globals.css";

export const metadata: Metadata = {{
  title: "{config.project_name}",
  description: "{config.description or "Next.js application"}",
}};

export default function RootLayout({{
  children,
}}: {{
  children: React.ReactNode;
}}) {{
  return (
    <html lang="en">
      <body>{{children}}</body>
    </html>
  );
}}
'''
        ))

        # app/page.tsx
        files.append(TemplateFile(
            path="app/page.tsx",
            content=f'''export default function Home() {{
  return (
    <main className="container">
      <h1>{config.project_name}</h1>
      <p>{config.description or "Welcome to your Next.js application"}</p>
    </main>
  );
}}
'''
        ))

        # app/globals.css
        files.append(TemplateFile(
            path="app/globals.css",
            content='''* {
  margin: 0;
  padding: 0;
  box-sizing: border-box;
}

body {
  font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif;
}

.container {
  max-width: 1200px;
  margin: 0 auto;
  padding: 2rem;
}
'''
        ))

        # README.md
        files.append(TemplateFile(
            path="README.md",
            content=f'''# {config.project_name}

{config.description or "Next.js application"}

## Quick Start

```bash
npm install
npm run dev
```

Open [http://localhost:3000](http://localhost:3000)
'''
        ))

        return files


class ViteReactTemplate(ProjectTemplate):
    """Vite + React application template"""

    @property
    def name(self) -> str:
        return "vite-react"

    @property
    def display_name(self) -> str:
        return "Vite + React"

    @property
    def description(self) -> str:
        return "React application with Vite and TypeScript"

    @property
    def language(self) -> str:
        return "typescript"

    @property
    def tags(self) -> List[str]:
        return ["web", "react", "spa", "vite"]

    def generate(self, config: TemplateConfig) -> List[TemplateFile]:
        files = []

        # package.json
        package = {
            "name": config.project_name,
            "version": config.version,
            "type": "module",
            "scripts": {
                "dev": "vite",
                "build": "tsc && vite build",
                "preview": "vite preview",
                "lint": "eslint .",
            },
            "dependencies": {
                "react": "^18.0.0",
                "react-dom": "^18.0.0",
            },
            "devDependencies": {
                "@types/react": "^18.0.0",
                "@types/react-dom": "^18.0.0",
                "@vitejs/plugin-react": "^4.0.0",
                "typescript": "^5.0.0",
                "vite": "^5.0.0",
            }
        }
        files.append(TemplateFile(
            path="package.json",
            content=json.dumps(package, indent=2)
        ))

        # vite.config.ts
        files.append(TemplateFile(
            path="vite.config.ts",
            content='''import { defineConfig } from "vite";
import react from "@vitejs/plugin-react";

export default defineConfig({
  plugins: [react()],
});
'''
        ))

        # tsconfig.json
        tsconfig = {
            "compilerOptions": {
                "target": "ES2020",
                "useDefineForClassFields": True,
                "lib": ["ES2020", "DOM", "DOM.Iterable"],
                "module": "ESNext",
                "skipLibCheck": True,
                "moduleResolution": "bundler",
                "allowImportingTsExtensions": True,
                "resolveJsonModule": True,
                "isolatedModules": True,
                "noEmit": True,
                "jsx": "react-jsx",
                "strict": True,
                "noUnusedLocals": True,
                "noUnusedParameters": True,
                "noFallthroughCasesInSwitch": True
            },
            "include": ["src"],
            "references": [{"path": "./tsconfig.node.json"}]
        }
        files.append(TemplateFile(
            path="tsconfig.json",
            content=json.dumps(tsconfig, indent=2)
        ))

        # tsconfig.node.json
        files.append(TemplateFile(
            path="tsconfig.node.json",
            content=json.dumps({
                "compilerOptions": {
                    "composite": True,
                    "skipLibCheck": True,
                    "module": "ESNext",
                    "moduleResolution": "bundler",
                    "allowSyntheticDefaultImports": True
                },
                "include": ["vite.config.ts"]
            }, indent=2)
        ))

        # index.html
        files.append(TemplateFile(
            path="index.html",
            content=f'''<!DOCTYPE html>
<html lang="en">
  <head>
    <meta charset="UTF-8" />
    <meta name="viewport" content="width=device-width, initial-scale=1.0" />
    <title>{config.project_name}</title>
  </head>
  <body>
    <div id="root"></div>
    <script type="module" src="/src/main.tsx"></script>
  </body>
</html>
'''
        ))

        # src/main.tsx
        files.append(TemplateFile(
            path="src/main.tsx",
            content='''import React from "react";
import ReactDOM from "react-dom/client";
import App from "./App";
import "./index.css";

ReactDOM.createRoot(document.getElementById("root")!).render(
  <React.StrictMode>
    <App />
  </React.StrictMode>
);
'''
        ))

        # src/App.tsx
        files.append(TemplateFile(
            path="src/App.tsx",
            content=f'''import {{ useState }} from "react";

function App() {{
  const [count, setCount] = useState(0);

  return (
    <div className="app">
      <h1>{config.project_name}</h1>
      <p>{config.description or "Vite + React application"}</p>
      <button onClick={{() => setCount((c) => c + 1)}}>
        Count: {{count}}
      </button>
    </div>
  );
}}

export default App;
'''
        ))

        # src/index.css
        files.append(TemplateFile(
            path="src/index.css",
            content=''':root {
  font-family: Inter, system-ui, sans-serif;
  line-height: 1.5;
}

body {
  margin: 0;
  min-height: 100vh;
  display: flex;
  justify-content: center;
  align-items: center;
}

.app {
  text-align: center;
}

button {
  padding: 0.5rem 1rem;
  font-size: 1rem;
  cursor: pointer;
}
'''
        ))

        # README.md
        files.append(TemplateFile(
            path="README.md",
            content=f'''# {config.project_name}

{config.description or "Vite + React application"}

## Quick Start

```bash
npm install
npm run dev
```

Open [http://localhost:5173](http://localhost:5173)
'''
        ))

        return files
