"""
Python project templates - FastAPI, Flask, Django, CLI
"""
from typing import List, Dict
from ucts.templates.base import ProjectTemplate, TemplateFile, TemplateConfig


class FastAPITemplate(ProjectTemplate):
    """FastAPI REST API template"""

    @property
    def name(self) -> str:
        return "fastapi"

    @property
    def display_name(self) -> str:
        return "FastAPI"

    @property
    def description(self) -> str:
        return "Modern async REST API with FastAPI"

    @property
    def language(self) -> str:
        return "python"

    @property
    def tags(self) -> List[str]:
        return ["api", "rest", "async", "web"]

    @property
    def dependencies(self) -> Dict[str, List[str]]:
        return {
            "python": [
                "fastapi>=0.109.0",
                "uvicorn[standard]>=0.27.0",
                "pydantic>=2.0.0",
                "pydantic-settings>=2.0.0",
            ]
        }

    @property
    def dev_dependencies(self) -> Dict[str, List[str]]:
        return {
            "python": [
                "pytest>=8.0.0",
                "pytest-asyncio>=0.23.0",
                "httpx>=0.26.0",
                "black>=24.0.0",
                "ruff>=0.1.0",
            ]
        }

    def generate(self, config: TemplateConfig) -> List[TemplateFile]:
        project = config.project_name
        files = []

        # main.py
        files.append(TemplateFile(
            path="main.py",
            content=f'''"""
{project} - FastAPI Application
"""
from fastapi import FastAPI
from fastapi.middleware.cors import CORSMiddleware

from app.api import router
from app.config import settings

app = FastAPI(
    title="{project}",
    description="{config.description or 'API built with FastAPI'}",
    version="{config.version}",
)

# CORS
app.add_middleware(
    CORSMiddleware,
    allow_origins=settings.cors_origins,
    allow_credentials=True,
    allow_methods=["*"],
    allow_headers=["*"],
)

# Include routers
app.include_router(router, prefix="/api")


@app.get("/health")
async def health_check():
    """Health check endpoint"""
    return {{"status": "healthy", "version": "{config.version}"}}


if __name__ == "__main__":
    import uvicorn
    uvicorn.run("main:app", host="0.0.0.0", port=8000, reload=True)
'''
        ))

        # app/__init__.py
        files.append(TemplateFile(
            path="app/__init__.py",
            content='"""App package"""'
        ))

        # app/config.py
        files.append(TemplateFile(
            path="app/config.py",
            content='''"""
Application configuration
"""
from pydantic_settings import BaseSettings
from typing import List


class Settings(BaseSettings):
    """Application settings"""
    app_name: str = "{project}"
    debug: bool = False
    cors_origins: List[str] = ["*"]
    database_url: str = "sqlite:///./app.db"

    class Config:
        env_file = ".env"


settings = Settings()
'''.format(project=project)
        ))

        # app/api/__init__.py
        files.append(TemplateFile(
            path="app/api/__init__.py",
            content='''"""API routes"""
from fastapi import APIRouter

router = APIRouter()


@router.get("/")
async def root():
    """API root"""
    return {"message": "Welcome to the API"}
'''
        ))

        # app/models/__init__.py
        files.append(TemplateFile(
            path="app/models/__init__.py",
            content='"""Data models"""'
        ))

        # app/schemas/__init__.py
        files.append(TemplateFile(
            path="app/schemas/__init__.py",
            content='"""Pydantic schemas"""'
        ))

        # tests/test_main.py
        files.append(TemplateFile(
            path="tests/test_main.py",
            content='''"""Tests for main application"""
import pytest
from httpx import AsyncClient, ASGITransport
from main import app


@pytest.mark.asyncio
async def test_health_check():
    """Test health check endpoint"""
    async with AsyncClient(transport=ASGITransport(app=app), base_url="http://test") as client:
        response = await client.get("/health")
    assert response.status_code == 200
    assert response.json()["status"] == "healthy"


@pytest.mark.asyncio
async def test_api_root():
    """Test API root endpoint"""
    async with AsyncClient(transport=ASGITransport(app=app), base_url="http://test") as client:
        response = await client.get("/api/")
    assert response.status_code == 200
'''
        ))

        # tests/conftest.py
        files.append(TemplateFile(
            path="tests/conftest.py",
            content='''"""Pytest configuration"""
import pytest


@pytest.fixture
def anyio_backend():
    return "asyncio"
'''
        ))

        # requirements.txt
        deps = self.dependencies["python"]
        files.append(TemplateFile(
            path="requirements.txt",
            content='\n'.join(deps) + '\n'
        ))

        # requirements-dev.txt
        dev_deps = self.dev_dependencies["python"]
        files.append(TemplateFile(
            path="requirements-dev.txt",
            content='-r requirements.txt\n' + '\n'.join(dev_deps) + '\n'
        ))

        # .env.example
        files.append(TemplateFile(
            path=".env.example",
            content='''# Application settings
DEBUG=true
DATABASE_URL=sqlite:///./app.db
CORS_ORIGINS=["http://localhost:3000"]
'''
        ))

        # pyproject.toml
        files.append(TemplateFile(
            path="pyproject.toml",
            content=f'''[project]
name = "{project}"
version = "{config.version}"
description = "{config.description}"
readme = "README.md"
requires-python = ">=3.9"

[tool.pytest.ini_options]
asyncio_mode = "auto"
testpaths = ["tests"]

[tool.ruff]
line-length = 100
target-version = "py39"

[tool.black]
line-length = 100
target-version = ["py39"]
'''
        ))

        # README.md
        files.append(TemplateFile(
            path="README.md",
            content=f'''# {project}

{config.description or "FastAPI application"}

## Quick Start

```bash
# Create virtual environment
python -m venv venv
source venv/bin/activate  # Linux/Mac
# or: venv\\Scripts\\activate  # Windows

# Install dependencies
pip install -r requirements-dev.txt

# Run development server
uvicorn main:app --reload

# Run tests
pytest
```

## API Documentation

Once running, visit:
- Swagger UI: http://localhost:8000/docs
- ReDoc: http://localhost:8000/redoc
'''
        ))

        return files


class FlaskTemplate(ProjectTemplate):
    """Flask web application template"""

    @property
    def name(self) -> str:
        return "flask"

    @property
    def display_name(self) -> str:
        return "Flask"

    @property
    def description(self) -> str:
        return "Flask web application with Blueprints"

    @property
    def language(self) -> str:
        return "python"

    @property
    def tags(self) -> List[str]:
        return ["web", "api", "flask"]

    @property
    def dependencies(self) -> Dict[str, List[str]]:
        return {
            "python": [
                "flask>=3.0.0",
                "python-dotenv>=1.0.0",
            ]
        }

    def generate(self, config: TemplateConfig) -> List[TemplateFile]:
        project = config.project_name
        files = []

        # app.py
        files.append(TemplateFile(
            path="app.py",
            content=f'''"""
{project} - Flask Application
"""
from flask import Flask
from dotenv import load_dotenv

load_dotenv()


def create_app():
    """Application factory"""
    app = Flask(__name__)
    app.config.from_prefixed_env()

    # Register blueprints
    from routes import main_bp
    app.register_blueprint(main_bp)

    return app


app = create_app()

if __name__ == "__main__":
    app.run(debug=True)
'''
        ))

        # routes.py
        files.append(TemplateFile(
            path="routes.py",
            content='''"""Application routes"""
from flask import Blueprint, jsonify

main_bp = Blueprint("main", __name__)


@main_bp.route("/")
def index():
    """Home page"""
    return jsonify(message="Welcome to the API")


@main_bp.route("/health")
def health():
    """Health check"""
    return jsonify(status="healthy")
'''
        ))

        # requirements.txt
        files.append(TemplateFile(
            path="requirements.txt",
            content='\n'.join(self.dependencies["python"]) + '\n'
        ))

        # .env.example
        files.append(TemplateFile(
            path=".env.example",
            content='FLASK_DEBUG=true\nFLASK_SECRET_KEY=change-me\n'
        ))

        # README.md
        files.append(TemplateFile(
            path="README.md",
            content=f'''# {project}

{config.description or "Flask application"}

## Quick Start

```bash
pip install -r requirements.txt
flask run
```
'''
        ))

        return files


class DjangoTemplate(ProjectTemplate):
    """Django web application template"""

    @property
    def name(self) -> str:
        return "django"

    @property
    def display_name(self) -> str:
        return "Django"

    @property
    def description(self) -> str:
        return "Django web application with REST framework"

    @property
    def language(self) -> str:
        return "python"

    @property
    def tags(self) -> List[str]:
        return ["web", "api", "django", "orm"]

    @property
    def dependencies(self) -> Dict[str, List[str]]:
        return {
            "python": [
                "django>=5.0.0",
                "djangorestframework>=3.14.0",
                "python-dotenv>=1.0.0",
            ]
        }

    def generate(self, config: TemplateConfig) -> List[TemplateFile]:
        project = config.project_name.replace("-", "_")
        files = []

        # manage.py
        files.append(TemplateFile(
            path="manage.py",
            content=f'''#!/usr/bin/env python
"""Django management script"""
import os
import sys

if __name__ == "__main__":
    os.environ.setdefault("DJANGO_SETTINGS_MODULE", "{project}.settings")
    from django.core.management import execute_from_command_line
    execute_from_command_line(sys.argv)
''',
            executable=True
        ))

        # project/settings.py
        files.append(TemplateFile(
            path=f"{project}/settings.py",
            content=f'''"""Django settings for {project}"""
from pathlib import Path
import os
from dotenv import load_dotenv

load_dotenv()

BASE_DIR = Path(__file__).resolve().parent.parent
SECRET_KEY = os.getenv("SECRET_KEY", "change-me-in-production")
DEBUG = os.getenv("DEBUG", "True").lower() == "true"
ALLOWED_HOSTS = os.getenv("ALLOWED_HOSTS", "localhost,127.0.0.1").split(",")

INSTALLED_APPS = [
    "django.contrib.admin",
    "django.contrib.auth",
    "django.contrib.contenttypes",
    "django.contrib.sessions",
    "django.contrib.messages",
    "django.contrib.staticfiles",
    "rest_framework",
]

MIDDLEWARE = [
    "django.middleware.security.SecurityMiddleware",
    "django.contrib.sessions.middleware.SessionMiddleware",
    "django.middleware.common.CommonMiddleware",
    "django.middleware.csrf.CsrfViewMiddleware",
    "django.contrib.auth.middleware.AuthenticationMiddleware",
    "django.contrib.messages.middleware.MessageMiddleware",
]

ROOT_URLCONF = "{project}.urls"

TEMPLATES = [
    {{
        "BACKEND": "django.template.backends.django.DjangoTemplates",
        "DIRS": [],
        "APP_DIRS": True,
        "OPTIONS": {{
            "context_processors": [
                "django.template.context_processors.debug",
                "django.template.context_processors.request",
                "django.contrib.auth.context_processors.auth",
                "django.contrib.messages.context_processors.messages",
            ],
        }},
    }},
]

DATABASES = {{
    "default": {{
        "ENGINE": "django.db.backends.sqlite3",
        "NAME": BASE_DIR / "db.sqlite3",
    }}
}}

STATIC_URL = "static/"
DEFAULT_AUTO_FIELD = "django.db.models.BigAutoField"

REST_FRAMEWORK = {{
    "DEFAULT_PERMISSION_CLASSES": [
        "rest_framework.permissions.AllowAny",
    ]
}}
'''
        ))

        # project/urls.py
        files.append(TemplateFile(
            path=f"{project}/urls.py",
            content=f'''"""URL configuration for {project}"""
from django.contrib import admin
from django.urls import path
from django.http import JsonResponse


def health_check(request):
    return JsonResponse({{"status": "healthy"}})


urlpatterns = [
    path("admin/", admin.site.urls),
    path("health/", health_check),
]
'''
        ))

        # project/__init__.py
        files.append(TemplateFile(
            path=f"{project}/__init__.py",
            content='"""Django project"""'
        ))

        # project/wsgi.py
        files.append(TemplateFile(
            path=f"{project}/wsgi.py",
            content=f'''"""WSGI config for {project}"""
import os
from django.core.wsgi import get_wsgi_application

os.environ.setdefault("DJANGO_SETTINGS_MODULE", "{project}.settings")
application = get_wsgi_application()
'''
        ))

        # requirements.txt
        files.append(TemplateFile(
            path="requirements.txt",
            content='\n'.join(self.dependencies["python"]) + '\n'
        ))

        # README.md
        files.append(TemplateFile(
            path="README.md",
            content=f'''# {config.project_name}

{config.description or "Django application"}

## Quick Start

```bash
pip install -r requirements.txt
python manage.py migrate
python manage.py runserver
```
'''
        ))

        return files


class CLITemplate(ProjectTemplate):
    """Python CLI application template"""

    @property
    def name(self) -> str:
        return "cli"

    @property
    def display_name(self) -> str:
        return "Python CLI"

    @property
    def description(self) -> str:
        return "Command-line application with Click"

    @property
    def language(self) -> str:
        return "python"

    @property
    def tags(self) -> List[str]:
        return ["cli", "command-line", "tool"]

    @property
    def dependencies(self) -> Dict[str, List[str]]:
        return {
            "python": [
                "click>=8.0.0",
                "rich>=13.0.0",
            ]
        }

    def generate(self, config: TemplateConfig) -> List[TemplateFile]:
        project = config.project_name.replace("-", "_")
        files = []

        # main CLI
        files.append(TemplateFile(
            path=f"{project}/cli.py",
            content=f'''"""
{config.project_name} - Command Line Interface
"""
import click
from rich.console import Console

console = Console()


@click.group()
@click.version_option(version="{config.version}")
def cli():
    """{config.description or 'CLI application'}"""
    pass


@cli.command()
@click.argument("name", default="World")
def hello(name):
    """Say hello"""
    console.print(f"[green]Hello, {{name}}![/green]")


@cli.command()
def info():
    """Show application info"""
    console.print(f"[bold]{config.project_name}[/bold] v{config.version}")


if __name__ == "__main__":
    cli()
'''
        ))

        # __init__.py
        files.append(TemplateFile(
            path=f"{project}/__init__.py",
            content=f'"""{{config.project_name}}"""\n__version__ = "{config.version}"'
        ))

        # __main__.py
        files.append(TemplateFile(
            path=f"{project}/__main__.py",
            content=f'from {project}.cli import cli\ncli()'
        ))

        # pyproject.toml
        files.append(TemplateFile(
            path="pyproject.toml",
            content=f'''[build-system]
requires = ["setuptools>=61.0"]
build-backend = "setuptools.build_meta"

[project]
name = "{config.project_name}"
version = "{config.version}"
description = "{config.description}"
requires-python = ">=3.9"
dependencies = [
    "click>=8.0.0",
    "rich>=13.0.0",
]

[project.scripts]
{config.project_name} = "{project}.cli:cli"
'''
        ))

        # requirements.txt
        files.append(TemplateFile(
            path="requirements.txt",
            content='\n'.join(self.dependencies["python"]) + '\n'
        ))

        # README.md
        files.append(TemplateFile(
            path="README.md",
            content=f'''# {config.project_name}

{config.description or "Command-line application"}

## Installation

```bash
pip install -e .
```

## Usage

```bash
{config.project_name} --help
{config.project_name} hello World
```
'''
        ))

        return files
